with Interfaces;

use Interfaces;

package FILO is

   type Blockdev_Length is range 0 .. Interfaces.Integer_64'Last;
   subtype Blockdev_Offset is Blockdev_Length range 0 .. Blockdev_Length'Last - 1;
   subtype Partition_Length is Blockdev_Length;
   subtype Partition_Offset is Blockdev_Offset;

   BLOCK_SIZE : constant := 512;
   type Block_Offset is range 0 .. Partition_Offset'Last / BLOCK_SIZE;

   subtype Index_Type is Natural range 0 .. Natural'Last - 1;
   subtype Index_Type16 is Index_Type range 0 .. Index_Type'Last - 1;
   subtype Index_Type32 is Index_Type16 range 0 .. Index_Type16'Last - 2;

   type Buffer_Type is array (Index_Type range <>) of Unsigned_8
   with
      Pack;

   function Read_LE16 (Buf : Buffer_Type; Off : Index_Type16) return Unsigned_16
   is
     (Shift_Left (Unsigned_16 (Buf (Buf'First + Off + 1)), 8) or
      Unsigned_16 (Buf (Buf'First + Off)))
   with
      Pre =>
         Buf'First <= Index_Type16'Last - Off and then
         Buf'First + Off + 1 <= Buf'Last;

   function Read_LE32 (Buf : Buffer_Type; Off : Index_Type32) return Unsigned_32
   is
     (Shift_Left (Unsigned_32 (Buf (Buf'First + Off + 3)), 24) or
      Shift_Left (Unsigned_32 (Buf (Buf'First + Off + 2)), 16) or
      Shift_Left (Unsigned_32 (Buf (Buf'First + Off + 1)),  8) or
      Unsigned_32 (Buf (Buf'First + Off)))
   with
      Pre =>
         Buf'First <= Index_Type32'Last - Off and then
         Buf'First + Off + 3 <= Buf'Last;

   generic
      type T is mod <>;
   function Gen_Is_Power_Of_2 (Val : T) return Boolean;
   function Gen_Is_Power_Of_2 (Val : T) return Boolean
   is
      (Val /= 0 and (Val and (Val - 1)) = 0);

   function Is_Power_Of_2 is new Gen_Is_Power_Of_2 (Unsigned_16);
   function Is_Power_Of_2 is new Gen_Is_Power_Of_2 (Unsigned_32);

   Space             : constant Character := ' ';
   Form_Feed         : constant Character := Character'Val (16#0c#);
   New_Line          : constant Character := Character'Val (16#0a#);
   Carriage_Return   : constant Character := Character'Val (16#0d#);
   Tab               : constant Character := Character'Val (16#09#);
   Vertical_Tab      : constant Character := Character'Val (16#0b#);

   function Is_Space (Ch : Character) return Boolean
   is
      (Ch = ' ' or (Tab <= Ch and Ch <= Carriage_Return));

   function Equal (Str : String; Buf : Buffer_Type) return Boolean
   with
      Pre => Str'Length = Buf'Length;

end FILO;
